# -*- coding: utf-8 -*-

u"""● TrM_PolyNormalFlatter (ver0.50)

ソフトエッジ法線 平坦化（クリーンアップ）ツール

▼ インストール

	一般的なPythonスクリプトと同じく、起動時にロードして使用します。
	
	1. mayaスクリプトフォルダにこのファイルを移動します。
        (Windows7 であれば Documents\maya\{バージョン番号}\ja_JP\scripts\ 等）
	2. スクリプトフォルダにuserSetup.pyが無い場合は新規に作成します。
		userSetup.pyは空のテキストファイル(.txt)を作成し、名前をuserSetup.pyに変更します。
		ここで、拡張子が非表示になっていると、userSetup.py.txtになってしまう事があるので注意してください。
		その場合、メモ帳などで開いて別名保存する時に拡張子を変更する事が出来ます。
	3. userSetup.py に以下のコードを追加します。
		import TrM_PolyNormalFlatter
	4. Mayaを起動し、以下のコマンドをPythonモードで実行します。
		TrM_PolyNormalFlatter.run()
	5. 必要があればコマンドをシェルフに登録してください。

▼ 使い方

	1. 予めハードエッジにしたい部分はハードエッジに設定しておいてください。
	2. 平面化したいフェースを選択しスクリプトを実行します。
	3. 選択したフェースの頂点法線がフェース法線と一致するよう設定されます。

	重要な点は選択したフェースが平坦化される反面、周囲の法線が歪むという事です。
	フェースを選択する時は「周囲より優先的に平面化したい面」を選択するように
	すると綺麗に法線を設定することができます。

	例) 機械系モデリングで平面フェースのみを選択し、
	　　ベベルによって作られた小さいフェースは選択しないようにする。
	　　→ソフトエッジでありながら、ハードエッジのようなキッチリ感を出すことが
	　　　できる。加えてベベル部は滑らかなシェーディングになる。

▼ その他

	作者: Yomogi
	Twitter: yomogi_k
	Blog: http://trtools.jp/

	対応Mayaバージョン: PyMELが使用可能なMaya。(標準ではMaya2011以上)

	利用規約:

	1. このスクリプトはフリーソフトです。
	   商業、個人問わず自由に利用して頂くことができます。

	2. このスクリプトを使用して発生したいかなる問題にも
	   作者は一切の感知をしませんので予めご了承ください。

	3. 以下の行為を禁止します。
	 ・無許可での再配布。
	 ・このスクリプトを自分が作ったと偽る行為。

"""

import pymel.core as pm
import time

# 設定項目ここから

PRINT_MESSAGE = True  # メッセージを表示するか (True or False)
PRINT_PROGRESS = True  # 進捗を表示するか (True or False)
PROGRESS_SPAN = 0.5  # 進捗の表示更新頻度 [s]

# 設定項目ここまで


def _printMsg(*msgs):

    if PRINT_MESSAGE:
        print " ".join(m if isinstance(m, unicode) else str(m) for m in msgs)


def _printProgress(num, maxNum):

    if PRINT_PROGRESS:
        global updateTime
        t = time.clock()
        if not num or num == maxNum or t - updateTime > PROGRESS_SPAN:
            updateTime = t
            print ("%" + str(len(str(maxNum))) + "d / %d") % (num, maxNum)


def run():

    # 選択中のフェースを取得
    faces = [s for s in pm.selected(typ='float3')
             if isinstance(s, pm.MeshFace)]
    if not faces:
        pm.warning(u"ポリゴンフェースを選択してください。")
        return

    # フェースをメッシュ毎にグループ化する
    meshes = {}
    setdefault = meshes.setdefault
    for f in faces:
        n = f.node()
        setdefault(n, []).append(f)

    # 全てのメッシュに対して
    for mesh, faceGroups in meshes.items():

        _printMsg(u"\n法線の平面化:", mesh)

        # 高速化の為予めメソッドを抽出しておく
        getFNormal = mesh.getPolygonNormal
        getFVNormal = mesh.getFaceVertexNormal

        # フェース法線を辞書に取得する
        faces = [f for fg in faceGroups for f in fg]
        faceNormals = {}
        getNormal = faceNormals.get
        for f in faces:
            fId = f.currentItemIndex()
            faceNormals[fId] = getFNormal(fId)

        # 選択を境界頂点へ変換
        vtxs = [v for vg in
                pm.polyListComponentConversion(faces, tv=True, bo=True)
                for v in pm.PyNode(vg)]

        # 全ての境界頂点に対して
        vBase = mesh.name() + ".vtx[%d]"
        vfBase = mesh.name() + ".vtxFace[%d][%d]"
        normals = []
        addNormal = normals.append
        num = len(vtxs)

        _printMsg(u"法線を解析しています。")

        for i, vtx in enumerate(vtxs):

            # 表示を更新
            _printProgress(i, num)

            # 頂点IDと隣接するフェースIDを取得
            vtxId = vtx.currentItemIndex()
            faceIds = vtx.connectedFaces().indices()

            # 同じ法線の頂点フェースをグループ化
            groups = {}
            setdefault = groups.setdefault
            for fId in faceIds:
                normal = getFVNormal(fId, vtxId)
                setdefault(normal.get(), []).append(fId)

            # 選択されているフェースの法線の平均を設定する
            lenGroups = len(groups)
            for fIds in groups.itervalues():
                sumVec = [0.0] * 3
                sumCount = 0
                for fId in fIds:

                    # フェースが選択されていれば
                    normal = getNormal(fId, None)
                    if normal:
                        sumVec[0] += normal[0]
                        sumVec[1] += normal[1]
                        sumVec[2] += normal[2]
                        sumCount += 1

                # 法線平均を求める
                if sumCount:
                    vec = (sumVec[0] / sumCount,
                           sumVec[1] / sumCount,
                           sumVec[2] / sumCount)

                    if lenGroups > 1:
                        # 頂点フェースに設定
                        vfs = [vfBase % (vtxId, fId) for fId in fIds]
                        addNormal((vfs, vec))
                    else:
                        # 頂点に設定
                        addNormal((vBase % vtxId, vec))

        _printProgress(num, num)
        _printMsg(u"法線を設定しています。")

        for i, n in enumerate(normals):
            _printProgress(i, num)
            pm.polyNormalPerVertex(n[0], xyz=n[1])

        _printProgress(num, num)
        _printMsg(u"完了しました。")
