# -*- coding: utf-8 -*-

u"""● TrM PolyNormalTools Ver 0.21

▼ 概要

    このパッケージは以下の２つのツールを含みます。
    
    ▽ 1. TrM 頂点平面化ツール
        フェース、頂点、法線を元に、自由に平面・移動方向を定義できます。
        定義された平面と移動方向を元に、複数の頂点を一度に平面上へ移動させることができます。
    
    ▽ 2. TrM 法線設定ツール
        フェース、頂点から法線を取得し、他のコンポーネントに再設定することができます。
        複数コンポーネントを選択した場合、その平均が取られます。
        また、取得する法線はワールド座標、ローカル座標を切り替えることができます。
        
        その他、補助ツールとして法線ディスプレイのON/OFF、法線のロック解除機能があります。
    
    ▽ インタフェース
        英語/日本語 の２言語に対応しています。
        
        英語に関してはかなりヘッポコな訳なので、間違っている所があればご連絡いただけると幸いです。
        というか、たぶんそこら中間違ってると思います。


▼ Description

    This package have 2 tools.
    
    ▽ 1. TrM Make Planar Tool
        First, you can define vertual plane from polygon selection in dialog.
        Next, make planar.
    
    ▽ 2. TrM Set Normal Tool
        First, you can capture normal from polygon selection.
        Next, you can set the normal to selected vertices.
    
    ▽ Language
        This tools are supporting for Japanese or English.
        The language is switch automatically.
        (But auther can't speak English very well. sorry.)
    
    
    Thank you.


▼ インストール:

    一般的なPythonスクリプトと同じく、起動時にロードして使用します。
    
    1. mayaスクリプトフォルダにこのファイルを移動します。
        (Windows7 であれば Documents\maya\{バージョン番号}\ja_JP\scripts\ 等）
        
    2. スクリプトフォルダにuserSetup.pyが無い場合は新規に作成します。
        userSetup.pyは空のテキストファイル(.txt)を作成し、名前をuserSetup.pyに変更します。
        ここで、拡張子が非表示になっていると、userSetup.py.txtになってしまう事があるので注意してください。
        その場合、メモ帳などで開いて別名保存する時に拡張子を変更する事が出来ます。
        
    3. userSetup.py に以下のコードを追加します。
        import TrM_PolyNormalTools
        
    4. Mayaを起動し、以下の各コマンドをPythonモードで実行します。
        TrM_PolyNormalTools.runMakePlanar()
        または
        TrM_PolyNormalTools.runSetNormal()
        
    5. 必要があれば各コマンドをシェルフに登録してください。
   

▼ Installation:

    You must load this script at Maya starting
    
    1. Copy this script to maya scripts folder.
    	ex) Documents/maya/{VERSION_NUMBER}/scripts/
        
    2. Make a userSetup.py file in the scripts folder if not exists the file.
    	(userSetup.py is a empty text file. Please google it if you need more infomation.)
        
    3. Add the follow code to userSetup.py.
        import TrM_PolyNormalTools
        
    4. Restart maya, and run the follow codes by python mode.
        TrM_PolyNormalTools.runMakePlanar()
        or
        TrM_PolyNormalTools.runSetNormal()
        
    5. Register the commands to your shelf.


▼ コマンドオプション:

    各コマンドはそれぞれオプション引数を持っています。

    runMakePlanar:
        printValue (Boolean): ベクトル情報の出力をON/OFFすることができます。デフォルトはTrueです。
        angleThreshold (float): 平面と移動方向が平行なとき、処理を失敗させるしきい値を設定します。
                               デフォルトは0.000001です。
    	例) runMakePlanar(printValue=False, angleThreshold=0.0001)
   
    runSetNormal:
        closeUtil (Boolean): 起動時に補助ツールを閉じた状態にするか設定します。デフォルトはFalseです。
   
    	例) runSetNormal(closeUtil=True)
 
 
 ▼ Command Options:
 
 	The commands have lunch options.
 	
    runMakePlanar:
        printValue (Boolean): Output the vector infomation. Default is True.
        angleThreshold (float): Threshold of cross operation between the plane normal and the move vector.
                                Default is 0.000001.
    	ex) runMakePlanar(printValue=False, angleThreshold=0.0001)
   
    runSetNormal:
        closeUtil (Boolean): Rollup the utility panel at the start. Default is False.
   
    	ex) runSetNormal(closeUtil=True)
   
▼ Supported Versions:
   2011 or highter

▼ Log:
   2013/02/17 SetNormalTool ver0.11
   2013/02/04 SetNormalTool ver0.1
   2013/02/11 MakePlanarTool ver0.21
   2013/01/29 MakePlanarTool ver0.2
   2011/04/11 MakePlanarTool ver0.1

Author: Yomogi

Twitter: yomogi_k

Blog: http://trtools.jp/

"""

import math
import pymel.core as pm
import pymel.core.nodetypes as nt


class Vector3(object):
    
    """Class of 3D vector."""
    
    def __init__(self, x=0.0, y=0.0, z=0.0, name=""):
        """Constructor
        
        Name is not inherit from other instance at the operation.
        
        Kwargs:
            x (Vector3 or list[Vector3]): X value. Unpack if it have list.
            y (Vector3): Y value.
            z (Vector3): Z value.
            name (string): The vector name.
        """
        
        if isinstance(x, list):
            z = x[2]
            y = x[1]
            x = x[0]
        self.x = float(x)
        self.y = float(y)
        self.z = float(z)
        self.name = str(name)
    
    def __add__(self, other):
        """Addision operator."""
        
        if isinstance(other, Vector3):
            f = Vector3(self.x + other.x, self.y + other.y, self.z + other.z)
            return f
        else:
            return Vector3(self.x + other, self.y + other, self.z + other)
    
    def __sub__(self, other):
        """Subtraction operator."""
        
        if isinstance(other, Vector3):
            return Vector3(self.x - other.x, self.y - other.y, self.z - other.z)
        else:
            return Vector3(self.x - other, self.y - other, self.z - other)
    
    def __mul__(self, other):
        """Multiplication operator."""
         
        if isinstance(other, Vector3):
            return Vector3(self.x * other.x, self.y * other.y, self.z * other.z)
        else:
            return Vector3(self.x * other, self.y * other, self.z * other)
    
    def __div__(self, other):
        """Division operator."""
        
        if isinstance(other, Vector3):
            return Vector3(self.x / other.x, self.y / other.y, self.z / other.z)
        else:
            return Vector3(self.x / other, self.y / other, self.z / other)
    
    def __str__(self):
        """Convert to string."""
        
        return "x=" + str(self.x) + ", y=" + str(self.y) + ", z=" + str(self.z)
    
    def inner(self, other):
        """Get inner product.
        
        Args:
            other (Vector3): Other instance.
        """
        
        return self.x * other.x + self.y * other.y + self.z * other.z
    
    def outer(self, other):
        """Get outer product.
        
        Args:
            other (Vector3): Other instance.
        """
        
        f = Vector3()
        f.x = self.y * other.z - self.z * other.y
        f.y = self.z * other.x - self.x * other.z
        f.z = self.x * other.y - self.y * other.x
        return f
    
    def getLength(self):
        """Get vector length."""
        
        length = math.sqrt(self.x * self.x + self.y * self.y + self.z * self.z)
        return length
    
    def getNormalized(self):
        """Get normalized vector."""
        
        length = self.getLength()
        if length != 0.0:
            length = 1.0 / length
        
        return self * length
    
    def getReversed(self):
        """Get reversed vector."""
        
        return Vector3(self.x * -1, self.y * -1, self.z * -1)
    
    def setList(self, sourceList):
        """Set vector values from list."""
        
        self.x = sourceList[0]
        self.y = sourceList[1]
        self.z = sourceList[2]
    
    def getList3(self):
        """Get list with three values."""
        
        return [self.x, self.y, self.z]
    
    def getList4(self):
        """Get list with four values."""
        
        return [self.x, self.y, self.z, 0.0]
    
    @staticmethod
    def getAverage(vectors):
        """Get average of vectors.
        
        Args:
            vectors (list[Vector3]): Collection of vector3.
            
        Returns:
            Vector3. The vector of average.
                     Return vector having zero, if the list of vector has not value..
        
        Raises:
            No raise.
        """
        
        vecSum = Vector3()
        if len(vectors) == 0:
            return vecSum
        
        for n in vectors:
            vecSum = vecSum + n
        
        return vecSum / len(vectors)
    
    @classmethod
    def fromPolygon(cls, vtx1, vtx2, vtx3):
        """Set a plane normal from three vertex positions.
        
        Args:
            vtx1 (Vector3): Position of the first vertex.
            vtx2 (Vector3): Position of the second vertex.
            vtx3 (Vector3): Position of the third vertex.
        """
        
        a = vtx2 - vtx1
        b = vtx3 - vtx1
        return a.outer(b).getNormalized()
    
    def toWorldSpace(self, m):
        """Convert vector to world space from local space.
        
        Args:
            m (list[float]): List of float containing matrix values.
        """
        m1 = Vector3(m[0:3])
        m2 = Vector3(m[4:7])
        m3 = Vector3(m[8:11])
        a1 = m2.outer(m3)
        a2 = m3.outer(m1)
        a3 = m1.outer(m2)
        vec = Vector3()
        vec.x = a1.x * self.x + a2.x * self.y + a3.x * self.z
        vec.y = a1.y * self.x + a2.y * self.y + a3.y * self.z
        vec.z = a1.z * self.x + a2.z * self.y + a3.z * self.z
        return vec.getNormalized()
    
    def toLocalSpace(self, m):
        """Convert vector to local space from world space.
        
        Args:
            m (list[float]): List of float containing matrix values.
        """
        m1 = Vector3(m[0:3])
        m2 = Vector3(m[4:7])
        m3 = Vector3(m[8:11])
        a1 = m2.outer(m3)
        a2 = m3.outer(m1)
        a3 = m1.outer(m2)
        vec = Vector3()
        vec.x = a1.x * self.x - a2.x * self.y - a3.x * self.z
        vec.y = a2.y * self.y - a1.y * self.x - a3.y * self.z
        vec.z = a3.z * self.z - a1.z * self.x - a2.z * self.y
        return vec.getNormalized()
    
    @classmethod
    def byLeastSquare(cls, vtxList, setNormal=True, setOffset=True):
        """Set the plane normal and offset by least square method.
        
        Args:
            vtxList (list[Vector3]): Collection of vertex positions.
        
        Kwargs:
            setNormal (bool): Does it set normal?
            setOffset (bool): Does it set offset?
        """
        
        vtxNum = len(vtxList)
        posMin = Vector3()
        posMax = Vector3()
        posAvg = Vector3()
        
        # Get bounding box and average position.  
        for v in vtxList:
            if posMin.x > v.x:
                posMin.x = v.x
            if posMin.y > v.y:
                posMin.y = v.y
            if posMin.z > v.z:
                posMin.z = v.z
            if posMax.x < v.x:
                posMax.x = v.x
            if posMax.y < v.y:
                posMax.y = v.y
            if posMax.z < v.z:
                posMax.z = v.z
            posAvg += v
        posAvg = posAvg / vtxNum
        
        # Get error from average.  
        errorVecs = []
        for v in vtxList:
            errorVecs.append(v - posAvg)
        posMin = posMin - posAvg
        posMax = posMax - posAvg
        
        # Calculate each sum values.  
        xx = 0.0
        yy = 0.0
        xy = 0.0
        xz = 0.0
        yz = 0.0
        x = 0.0
        y = 0.0
        z = 0.0
        for v in errorVecs:
            xx += (v.x ** 2)
            yy += (v.y ** 2)
            xy += (v.x * v.y)
            xz += (v.x * v.z)
            yz += (v.y * v.z)
            x += v.x
            y += v.y
            z += v.z
        
        # Calculate polygon.  
        d = xx * (y * y - vtxNum * yy) - \
            2 * xy * x * y + yy * x * x + vtxNum * xy * xy
        a = -1 * (xy * (y * z - vtxNum * yz) + x * (yz * y - yy * z) + \
            xz * (vtxNum * yy - y * y)) / d
        b = (xx * (y * z - vtxNum * yz) + x * (-1 * xy * z - xz * y) + \
            yz * x * x + vtxNum * xy * xz) / d
        c = (xx * (yz * y - yy * z) + xy * xy * z - xy * xz * y + \
            (yy * xz - xy * yz) * x) / d
        poly = [Vector3(posMin.x, 0, posMin.z),
                Vector3(posMin.x, 0, posMax.z),
                Vector3(posMax.x, 0, posMax.z)]
        
        for i in range(3):
            poly[i].y = (-1 * a * poly[i].x + poly[i].z - c) / b
            poly[i] = poly[i] + posAvg
        
        return cls.fromPolygon(poly[0], poly[1], poly[2])


class PolyHelper(object):
    '''
    Class of polygon edit helper.
    '''
    
    @classmethod
    def getSelectedTransforms(cls, forceConvert=False):
        """Get the selected all transform nodes.
        
        This method returns a list of Transform non-overlapping.
        And these Transform nodes are invariably have a mesh.
        """
        
        meshes = pm.selected(type="mesh")
        transforms = pm.selected(type="transform")
        if forceConvert:
            sel = pm.selected(type="float3")
            if len(sel) != 0:
                last = ""
                for s in sel:
                    name = str(s).split(".")[0]
                    if last == name:
                        continue
                    last = name
                    node = pm.PyNode(name)
                    if isinstance(node, nt.Mesh):
                        meshes.append(node)
        
        table = {}
        for trans in transforms:
            shape = trans.getShape()
            if isinstance(shape, nt.Mesh):
                name = str(trans)
                if not name in table:
                    table[name] = trans
        
        for mesh in meshes:
            trans = mesh.getParent()
            if not trans is None:
                name = str(trans)
                if not name in table:
                    table[name] = trans
        
        return table.values()
    
    @classmethod
    def __getSelected(cls, selMask, forceConvert, extractMesh, **keywords):
        
        # Get all selected components.
        if extractMesh:
            sel = pm.selected(type="float3")
        else:
            sel = pm.selected()
        if len(sel) != 0:
            # Conversion components.
            if forceConvert:
                keywords["toVertex"] = True
                sel = pm.polyListComponentConversion(sel, **keywords)
            if len(sel) != 0:
                # Expand components.
                sel = pm.filterExpand(sel, sm=selMask)
                if not sel is None:
                    return sel
        return []
    
    @classmethod
    def getSelectedVertices(cls, forceConvert=False, extractMesh=False, **keywords):
        """Get selected vertex in Maya.
        
        Kworgs:
            forceConvert (Boolean): Convert form other components.
            extractMesh (Boolean): Extract shape object from selected list.
            **keywords: polyListComponentConversion keywords.
        """
        keywords["toVertex"] = True
        return PolyHelper.__getSelected(31, forceConvert, extractMesh, **keywords)
    
    @classmethod
    def getSelectedEdges(cls, forceConvert=False, extractMesh=False, **keywords):
        """Get selected edge in Maya.
        
        Kworgs:
            forceConvert (Boolean): Convert form other components.
            extractMesh (Boolean): Extract shape object from selected list.
            **keywords: polyListComponentConversion keywords.
        """
        keywords["toEdge"] = True
        return PolyHelper.__getSelected(32, forceConvert, extractMesh, **keywords)
    
    @classmethod
    def getSelectedFaces(cls, forceConvert=False, extractMesh=False, **keywords):
        """Get selected face in Maya.
        
        Kworgs:
            forceConvert (Boolean): Convert form other components.
            extractMesh (Boolean): Extract shape object from selected list.
            **keywords: polyListComponentConversion keywords.
        """
        keywords["toFace"] = True
        return PolyHelper.__getSelected(34, forceConvert, extractMesh, **keywords)
    
    @classmethod
    def getSelectedUVs(cls, forceConvert=False, extractMesh=False, **keywords):
        """Get selected uv in Maya.
        
        Kworgs:
            forceConvert (Boolean): Convert form other components.
            extractMesh (Boolean): Extract shape object from selected list.
            **keywords: polyListComponentConversion keywords.
        """
        keywords["toUV"] = True
        return PolyHelper.__getSelected(35, forceConvert, extractMesh, **keywords)
    
    @classmethod
    def getSelectedVertexFaces(cls, forceConvert=False, extractMesh=False, **keywords):
        """Get selected vertex-face in Maya.
        
        Kworgs:
            forceConvert (Boolean): Convert form other components.
            extractMesh (Boolean): Extract shape object from selected list.
            **keywords: polyListComponentConversion keywords.
        """
        keywords["toVertexFace"] = True
        return PolyHelper.__getSelected(70, forceConvert, extractMesh, **keywords)
    
    @classmethod
    def getVertexPosition(cls, vertex):
        """Get world space positions from vertex list.
        
        Args:
            vertex (string): String of vertex name.
        """
        
        # Get world space position.
        pos = pm.xform(vertex, q=True, a=True, ws=True, t=True)
        vec = Vector3(pos)
        vec.name = vertex
        
        return vec
    
    @classmethod
    def getVertexPositions(cls, vtxList):
        """Get world space positions from vertex list.
        
        Args:
            vtxList (list[string]): Collection of vertex name string.
        """
        
        # Get world space position.
        vecs = []
        for vtx in vtxList:
            vecs.append(PolyHelper.getVertexPosition(vtx))
        
        return vecs
    
    
    @classmethod
    def moveVertex(cls, vector):
        """Move vertex in world space.
        
        Args:
            vector (Vector3): Position with the vertex name.
        
        Returns:
            Vector3: Vertex normal.
        """
        pm.xform(vector.name, a=True, ws=True, t=vector.getList3())
    
    
    @classmethod
    def getVertexNormal(cls, vertex, toLocal=False, toWorld=False, matrix=None):
        """Get vertex normal.
        
        Args:
            vertex (string): String of vertex name.
        """
        # Get the vertex normal.
        normals = pm.polyNormalPerVertex(vertex, q=True, xyz=True)
        normal = Vector3()
        num = len(normals) / 3
        for i in range(0, num):
            normal.x = normal.x + normals[3 * i]
            normal.y = normal.y + normals[3 * i + 1]
            normal.z = normal.z + normals[3 * i + 2]
        normal = normal / num
        
        if toWorld:
            normal = normal.toWorldSpace(matrix)
        elif toLocal:
            normal = normal.toLocalSpace(matrix)
        
        return normal.getNormalized()
    
    @classmethod
    def getVertexNormals(cls, vtxList, toLocal=False, toWorld=False):
        """Get vertex normals from vertex list.
        
        Args:
            vtxFaceList (list[string]): Collection of vertex name string.
        
        Returns:
            list[Vector3]: Collection of vertex normal.
        """
        normals = []
        matrix = None
        lastShape = ""
        for vtx in vtxList:
            if toLocal or toWorld:
                shape = vtx.split(".")[0]
                if lastShape != shape:
                    lastShape = shape
                    matrix = pm.xform(shape, q=True, m=True, ws=True)
            normals.append(PolyHelper.getVertexNormal(vtx, toLocal=toLocal, toWorld=toWorld, matrix=matrix))
        return normals
    
    @classmethod
    def getFaceNormal(cls, face, toLocal=False, toWorld=False, matrix=None):
        """Get selected face normals. 
        
        Args:
            face (string): String of face name.
        
        Returns:
            Vector3: Nace normal.
        """
        
        # Get the face normal.
        info = pm.polyInfo(face, faceNormals=True)[0]
        info = info.split(" ")
        normal = Vector3(float(info[-3]), float(info[-2]), float(info[-1]))
        
        if toWorld:
            normal = normal.toWorldSpace(matrix)
        elif toLocal:
            normal = normal.toLocalSpace(matrix)
        
        return normal
    
    
    @classmethod
    def getFaceNormals(cls, faceList, toLocal=False, toWorld=False):
        """Get face normals from face list.
        
        Args:
            vtxFaceList (list[string]): Collection of face name string.
        
        Returns:
            list[Vector3]: Collection of face normal.
        """
        normals = []
        matrix = None
        lastShape = ""
        for face in faceList:
            if toLocal or toWorld:
                shape = face.split(".")[0]
                if lastShape != shape:
                    lastShape = shape
                    matrix = pm.xform(shape, q=True, m=True, ws=True)
            normals.append(PolyHelper.getFaceNormal(face, toLocal=toLocal, toWorld=toWorld, matrix=matrix))
        return normals
    
    @classmethod
    def getVertexFaceNormal(cls, vtxFace, toLocal=False, toWorld=False, matrix=None):
        """Get vertex-face normal.
        
        Args:
            vtxFace (string): String of vertex-face name.
        
        Returns:
            Vector3: Vertex-face normal.
        """
        # Get the vertex normal.
        normal = pm.polyNormalPerVertex(vtxFace, q=True, xyz=True)
                
        if toWorld:
            normal = normal.toWorldSpace(matrix)
        elif toLocal:
            normal = normal.toLocalSpace(matrix)
        
        return Vector3(normal).getNormalized()
    
    @classmethod
    def getVertexFaceNormals(cls, vtxFaceList, toLocal=False, toWorld=False):
        """Get vertex-face normals from vertex-face list.
        
        Args:
            vtxFaceList (list[string]): Collection of vertex-face name string.
        
        Returns:
            list[Vector3]: Collection of vertex-face normal.
        """
        normals = []
        matrix = None
        lastShape = ""
        for vtxFace in vtxFaceList:
            if toLocal or toWorld:
                shape = vtxFace.split(".")[0]
                if lastShape != shape:
                    lastShape = shape
                    matrix = pm.xform(shape, q=True, m=True, ws=True)
            normals.append(PolyHelper.getVertexFaceNormal(vtxFace, toLocal=toLocal, toWorld=toWorld, matrix=matrix))
        return normals
    

class MakePlanarToolWindow(object):
    
    u'''
    Class of main window.
    '''
    
    jpText = {
        "title" : u"TrM 頂点平面化ツール",
        "planeNormal" : u"法線:",
        "planeOffset" : u"オフセット:",
        "moveVector" : u"ベクトル:",
        "xAxis" : u"X軸",
        "yAxis" : u"Y軸",
        "zAxis" : u"Z軸",
        "definePlane" : u"平面の定義",
        "defineMoveVector" : u"移動方向の定義",
        "getAll" : u"全てのパラメータを選択から取得",
        "reset" : u"リセット",
        "getSelection" : u"　選択から取得　",
        "runAndClose" : u"平面化",
        "run" : u"適用",
        "close" : u"閉じる",
        
        "getNormalError" : u"頂点へ変換可能なコンポーネントを選択してください。",
        "getOffsetError" : u"頂点へ変換可能なコンポーネントを選択してください。",
        "getVectorError" : u"1つのエッジか、2つの頂点を選択してください。",
        "runError" : u"コンポーネントを選択してください。",
        "crossError" : u"平面法線と移動ベクトルが直角に交差しているため移動できません。",
        "completed" : u"Completed!"}
    
    enText = {
        "title" : u"TrM MakePlanar Tool",
        "planeNormal" : u"Normal:",
        "planeOffset" : u"Offset:",
        "moveVector" : u"Vector:",
        "xAxis" : u"X Axis",
        "yAxis" : u"Y Axis",
        "zAxis" : u"Z Axis",
        "definePlane" : u"Define Plane",
        "defineMoveVector" : u"Define Move Vector",
        "getAll" : u"Get all parameters from selection",
        "reset" : u"Reset",
        "getSelection" : u"Get from selection",
        "runAndClose" : u"Apply and Close",
        "run" : u"Apply",
        "close" : u"Close",
        
        "getNormalError" : u"Please select component.",
        "getOffsetError" : u"Please select component.",
        "getVectorError" : u"Please select a edge or two vertices.",
        "runError" : u"Please select components.",
        "crossError" : u"The vertices can't move because vectors are crossing at right angle.",
        "completed" : u"Completed!"}
    
    def __init__(self, printValue=True, angleThreshold=0.000001):
        u'''
        Initialize main window.
        '''
        
        # Get the language table.
        lang = pm.about(uiLanguage=True)
        if lang == 'ja_JP':
            self.text = MakePlanarToolWindow.jpText
        else:
            self.text = MakePlanarToolWindow.enText
        
        self.mainWindow = pm.window(title=self.text["title"])
        self.floatFGrpPlaneNormal = None
        self.floatFGrpPlaneOffset = None
        self.floatFGrpMoveVector = None
        
        self.planeNormal = Vector3()
        self.planeOffset = Vector3()
        self.moveVector = Vector3()
        
        template = pm.uiTemplate()
        template.define(pm.frameLayout, mh=6, mw=6)
        template.define(pm.floatFieldGrp, numberOfFields=3, cw4=(72, 72, 72, 72))
        template.define(pm.columnLayout, adj=True, rs=2)
        
        with self.mainWindow:
            with pm.frameLayout(lv=False, bv=False, mh=2, mw=2):
                with template:
                    with pm.columnLayout():
                        with pm.frameLayout(lv=False):
                            with pm.columnLayout(rs=7):
                                with pm.horizontalLayout(ratios=(3, 1), spacing=4):
                                    pm.button(label=self.text["getAll"], height=36, c=pm.Callback(self.getAll))
                                    pm.button(label=self.text["reset"], height=26, c=pm.Callback(self.reset))
                                
                                # Plane section.
                                with pm.frameLayout(label=self.text["definePlane"]):
                                    with pm.columnLayout():
                                        with pm.rowLayout(numberOfColumns=3, cat=(1,"left", 75), cw3=(147, 72, 72)):
                                            pm.button(label=self.text["xAxis"], width=72, c=pm.Callback(self.setNormalAxis, 0))
                                            pm.button(label=self.text["yAxis"], width=72, c=pm.Callback(self.setNormalAxis, 1))
                                            pm.button(label=self.text["zAxis"], width=72, c=pm.Callback(self.setNormalAxis, 2))
                                        
                                        with pm.rowLayout(numberOfColumns=2, adj=2):
                                            self.floatFGrpPlaneNormal = pm.floatFieldGrp(label=self.text["planeNormal"],
                                                                                         cc=pm.Callback(self.floatFGrpPlaneNormal_ValueChanged))
                                            pm.button(label=self.text["getSelection"], c=pm.Callback(self.getPlaneNormal))
                                        with pm.rowLayout(numberOfColumns=2, adj=2):
                                            self.floatFGrpPlaneOffset = pm.floatFieldGrp(label=self.text["planeOffset"],
                                                                                         cc=pm.Callback(self.floatFGrpPlaneOffset_ValueChanged))
                                            pm.button(label=self.text["getSelection"], c=pm.Callback(self.getPlaneOffset))
                                        
                                # Move vector section.
                                with pm.frameLayout(label=self.text["defineMoveVector"]):
                                    with pm.columnLayout():
                                        with pm.rowLayout(numberOfColumns=3, cat=(1,"left", 75), cw3=(147, 72, 72)):
                                            pm.button(label=self.text["xAxis"], width=72, c=pm.Callback(self.setMoveVectorAxis, 0))
                                            pm.button(label=self.text["yAxis"], width=72, c=pm.Callback(self.setMoveVectorAxis, 1))
                                            pm.button(label=self.text["zAxis"], width=72, c=pm.Callback(self.setMoveVectorAxis, 2))
                                        
                                        with pm.rowLayout(numberOfColumns=2, adj=2):
                                            self.floatFGrpMoveVector = pm.floatFieldGrp(label=self.text["moveVector"],
                                                                                        cc=pm.Callback(self.floatFGrpMoveVector_ValueChanged))
                                            pm.button(label=self.text["getSelection"], c=pm.Callback(self.getMoveVector))
                                        
                        with pm.horizontalLayout(spacing=4, height=36):
                            pm.button(label=self.text["runAndClose"], c=pm.Callback(self.runAndClose))
                            pm.button(label=self.text["run"], c=pm.Callback(self.run))
                            pm.button(label=self.text["close"], c=pm.Callback(self.close))
        
        # Initialize vectors.
        self.printValue = False
        self.reset()
        self.printValue = printValue
        self.angleThreshold = angleThreshold
        
    
    # ▼ 全ての設定を選択から取得
    def getAll(self):
        
        if self.getPlaneNormal():
            self.getPlaneOffset()
            self.moveVector = Vector3(self.planeNormal.getList3())
            
            # Set GUI values.
            self._setFloatGrpValues(self.floatFGrpMoveVector, self.moveVector.getList3())
            return True
        else:
            return False
    
    # ▼ 平面法線を選択から取得
    def getPlaneNormal(self):
        
        # Select algorithm.
        faces = PolyHelper.getSelectedFaces()
        normals = PolyHelper.getFaceNormals(faces, toWorld=True)
        if len(normals) == 0:
            vecs = PolyHelper.getSelectedVertices(forceConvert=True)
            # Face no selected.
            if len(vecs) == 0:
                # No selected.
                pm.warning(self.text["getNormalError"])
                return False
            elif len(vecs) == 3:
                # 3 vertices selected.
                vecs = PolyHelper.getVertexPositions(vecs[0:3])
                self.planeNormal = Vector3.fromPolygon(vecs[0], vecs[1], vecs[2])
            else:
                # Other than 3 vertices selected.
                normals = PolyHelper.getVertexNormals(vecs, toWorld=True)
                self.planeNormal = Vector3.getAverage(normals).getNormalized()
        else:
            # Face selected.
            self.planeNormal = Vector3.getAverage(normals).getNormalized()
        
        
        # Set GUI values.
        self._setFloatGrpValues(self.floatFGrpPlaneNormal, self.planeNormal.getList3())
        return True
        
    
    # ▼ 平面オフセットを選択から取得
    def getPlaneOffset(self):
        vtxs = PolyHelper.getSelectedVertices(forceConvert=True)
        vecs = PolyHelper.getVertexPositions(vtxs)
        if len(vecs) == 0:
            pm.warning(self.text["getOffsetError"])
            return False
        
        # Get offset
        self.planeOffset = Vector3.getAverage(vecs)
        
        # Set GUI values.
        self._setFloatGrpValues(self.floatFGrpPlaneOffset, self.planeOffset.getList3())
        return True
    
    # ▼ 移動ベクトルを選択から取得
    def getMoveVector(self):
        vtxs = PolyHelper.getSelectedVertices(forceConvert=True)
        vecs = PolyHelper.getVertexPositions(vtxs)
        if len(vecs) != 2:
            pm.warning(self.text["getVectorError"])
            return False
        
        # Set move vector.
        self.moveVector = (vecs[0] - vecs[1]).getNormalized()
        
        # Set GUI values.
        self._setFloatGrpValues(self.floatFGrpMoveVector, self.moveVector.getList3())
        return True
    
    # ▼ 平面法線をXYZ軸で設定
    def setNormalAxis(self, axisIndex):
        vec = [0.0] * 3
        vec[axisIndex] = 1.0
        self.planeNormal = Vector3(vec)
        self._setFloatGrpValues(self.floatFGrpPlaneNormal, vec)
    
    # ▼ 移動ベクトルをXYZ軸で設定
    def setMoveVectorAxis(self, axisIndex):
        vec = [0.0] * 3
        vec[axisIndex] = 1.0
        self.moveVector = Vector3(vec)
        self._setFloatGrpValues(self.floatFGrpMoveVector, vec)
    
    # ▼ 全ての値をリセット
    def reset(self):
        """Reset all values."""
        
        # Set plane offset.
        self.planeOffset = Vector3()
        self._setFloatGrpValues(self.floatFGrpPlaneOffset, self.planeOffset.getList3())
        
        self.setNormalAxis(0) # 0 => X Axis
        self.setMoveVectorAxis(0) # 0 => X Axis
    
    # ▼ 平面化を実行しウィンドウを閉じる
    def runAndClose(self):
        if self.run():
            self.close()
    
    # ▼ 平面化を実行する
    def run(self):
        vtxs = PolyHelper.getSelectedVertices(forceConvert=True)
        vecs = PolyHelper.getVertexPositions(vtxs)
        if len(vecs) == 0:
            pm.warning(self.text["runError"])
            return
        
        # Get normalized vectors.
        pn = self.planeNormal.getNormalized()
        mv = self.moveVector.getNormalized()
        
        # Run make planar
        moveAngle = pn.inner(mv)
        if math.fabs(moveAngle) <= self.angleThreshold:
            pm.warning(self.text["crossError"])
            return False
        
        for vtx in vecs:
            offsetVec = self.planeOffset - vtx
            offsetAngle = offsetVec.inner(pn)
            distance = offsetAngle / moveAngle
            v = vtx + mv * distance
            v.name = vtx.name
            
            PolyHelper.moveVertex(v)
        
        print self.text["completed"]
        return True
    
    # ▼ ウィンドウを閉じる
    def close(self):
        self.mainWindow.delete()
    
    # ▼ PlaneNormal floatField が変更された時の処理
    def floatFGrpPlaneNormal_ValueChanged(self):
        new = self.floatFGrpPlaneNormal.getValue()
        self.planeNormal = Vector3(new)
        self._setFloatGrpValues(self.floatFGrpPlaneNormal, new)
        
    # ▼ PlaneOffset floatField が変更された時の処理
    def floatFGrpPlaneOffset_ValueChanged(self):
        new = self.floatFGrpPlaneOffset.getValue()
        self.planeOffset = Vector3(new)
        self._setFloatGrpValues(self.floatFGrpPlaneOffset, new)
        
    # ▼ MoveVector floatField が変更された時の処理
    def floatFGrpMoveVector_ValueChanged(self):
        new = self.floatFGrpMoveVector.getValue()
        self.moveVector = Vector3(new)
        self._setFloatGrpValues(self.floatFGrpMoveVector, new)
    
    # ▼ floatFieldGrpへベクトルを適応する
    def _setFloatGrpValues(self, floatGrp, values):
        """Set floatFieldGrp values, and adjustment precision.
        
        Args:
            values (list[float]): Collection of float.
        """
        
        # Get precision length.
        maxPre = 15
        maxView = 7
        pre = 1
        
        for val in values:
            s = str(val)
            s = s.split(".")
            if len(s) > 1:
                r = len(s[-1])
                if r > pre:
                    pre = r
        
        if len(values) == 3:
            values.append(0.0)
        
        # The precision is up to 7.
        if pre > maxView:
            pre = maxView
        
        floatGrp.precision(val=pre)
        floatGrp.setValue(values)
        floatGrp.precision(val=maxPre)
        
        if self.printValue:
            print "%s x=%s, y=%s, z=%s" % (floatGrp.getLabel(), 
                                             values[0], values[1], values[2])


class SetNormalToolWindow(object):

    jpText = {
        "title" : u"TrM 法線設定ツール",
        "runAndClose" : u"法線の設定",
        "run" : u"適用",
        "close" : u"閉じる",
        
        "utils" : u"補助",
        "dispNormal" : u"法線表示:",
        "normalSize" : u"法線サイズ:",
        "dispOff" : u"非表示",
        "dispVtx" : u"頂点",
        "dispFace" : u"フェース",
        "dispVtxFace" : u"頂点フェース",
        
        "setDispSize" : u"設定",
        "unlockNormal" : u"法線のロック解除",
        
        "settings" : u"設定",
        "worldSpace" : u"ワールドスペース",
        "localSpace" : u"ローカルスペース",
        "getSelection" : u"選択から取得",
        "axis" : u"軸:",
        "x" : u"X",
        "y" : u"Y",
        "z" : u"Z",
        "xValue" : u"X値:",
        "yValue" : u"Y値:",
        "zValue" : u"Z値:",
        "normalize" : u"法線の正規化",
        "reverse" : u"法線の反転",
        
        "noSelectedError" : u"コンポーネントが選択されていません。",
        
        "completed" : u"Completed!"}
    
    enText = {
        "title" : u"TrM SetNormal Tool",
        "runAndClose" : u"Apply and Close",
        "run" : u"Apply",
        "close" : u"Close",
        
        "utils" : u"Utility",
        "dispNormal" : u"Display Normal:",
        "normalSize" : u"Normal Size:",
        "dispOff" : u"OFF",
        "dispVtx" : u"Vertex",
        "dispFace" : u"Face",
        "dispVtxFace" : u"VertexFace",

        "setDispSize" : u"Set",
        "unlockNormal" : u"Unlock Normal",
        
        "settings" : u"Settings",
        "worldSpace" : u"World Space",
        "localSpace" : u"Local Space",
        "getSelection" : u"Get From Selection",
        "axis" : u"Axis:",
        "x" : u"X",
        "y" : u"Y",
        "z" : u"Z",
        "xValue" : u"X Value:",
        "yValue" : u"Y Value:",
        "zValue" : u"Z Value:",
        "normalize" : u"Normalize",
        "reverse" : u"Reverse",
        
        "noSelectedError" : u"Please select component.",
        
        "completed" : u"Completed!"}
    
    def __init__(self, closeUtil=False):
        '''
        Constructor
        '''
        # Get language.
        lang = pm.about(uiLanguage=True)
        if lang == 'ja_JP':
            self.text = SetNormalToolWindow.jpText
        else:
            self.text = SetNormalToolWindow.enText
        
        self.mainWindow = None
        self.utilFrame = None
        self.floatSGrpNmlDisp = None
        self.rdiBtnWorld = None
        self.rdiBtnLocal = None
        self.floatSGrpX = None
        self.floatSGrpY = None
        self.floatSGrpZ = None
        self.vector = Vector3()
        
        template = pm.uiTemplate()
        template.define(pm.frameLayout, mh=6, mw=6)
        template.define(pm.columnLayout, adj=True, rs=2)
        template.define(pm.separator, style="none", height=4)
        
        tempDisp = pm.uiTemplate()
        tempDisp.define(pm.separator, style="none", height=6)
        tempDisp.define(pm.button, width=80)
        
        xyzTemp = pm.uiTemplate()
        xyzTemp.define(pm.floatSliderGrp, field=True, min=-1.0, max=1.0, fmn=-1.0, fmx=1.0, cw3=(64, 72, 240), step=0.01, pre=5)
        
        with pm.window(title=self.text["title"]) as self.mainWindow:
            with pm.frameLayout(lv=False, bv=False, mh=2, mw=2):
                with template:
                    with pm.columnLayout():
                        with pm.frameLayout(lv=False):
                            with pm.columnLayout(rs=7):
                                with pm.frameLayout(l=self.text["utils"], cll=True, cl=closeUtil,
                                                     cc=pm.Callback(self.utilFrame_CollapseChanged)) as self.utilFrame:
                                    with pm.columnLayout(rs=0):
                                        with tempDisp:
                                            with pm.rowLayout(cw5=(90, 80, 80, 80, 80), adj=5, nc=5, cat=(1, "right", 0)):
                                                pm.text(l=self.text["dispNormal"])
                                                pm.button(l=self.text["dispVtx"], c=pm.Callback(self.buttonNormalDisplay_Clicked, 2))                                      
                                                pm.button(l=self.text["dispFace"], c=pm.Callback(self.buttonNormalDisplay_Clicked, 1))
                                                pm.button(l=self.text["dispVtxFace"], c=pm.Callback(self.buttonNormalDisplay_Clicked, 3))
                                                pm.button(l=self.text["dispOff"], c=pm.Callback(self.buttonNormalDisplay_Clicked, 0))
                                            pm.separator()
                                            with pm.rowLayout(nc=2, adj=2):
                                                self.floatSGrpNmlDisp = pm.floatSliderGrp(
                                                      l=self.text["normalSize"], min=0.01, max=5.0, fmn=0.000, fmx=100.0,
                                                      v=0.4, step=0.01, field=True, pre=3, cw3=(90, 48, 188))
                                                pm.button(l=self.text["setDispSize"], c=pm.Callback(self.buttonNormalSize_Clicked))
                                            pm.separator()
                                            pm.button(l=self.text["unlockNormal"], c=pm.Callback(self.buttonLockNormal_Clicked, False))
                                            pm.separator(h=1)
                                with pm.frameLayout(l=self.text["settings"]):
                                    with pm.columnLayout():
                                        pm.radioCollection()
                                        with pm.horizontalLayout(spacing=6):
                                            self.rdiBtnWorld = pm.radioButton(l=self.text["worldSpace"], select=True)
                                            self.rdiBtnLocal = pm.radioButton(l=self.text["localSpace"])
                                        pm.button(l=self.text["getSelection"], height=36, c=pm.Callback(self.buttonGetNormal_Clicked))
                                        pm.separator()
                                        with pm.rowLayout(numberOfColumns=4, cat=(1,"right", 0), cw4=(64, 72, 72, 72)):
                                            pm.text(l=self.text["axis"])
                                            pm.button(l=self.text["x"], w=72, c=pm.Callback(self.buttonAxis_Clicked, 0))
                                            pm.button(l=self.text["y"], w=72, c=pm.Callback(self.buttonAxis_Clicked, 1))
                                            pm.button(l=self.text["z"], w=72, c=pm.Callback(self.buttonAxis_Clicked, 2))
                                        with xyzTemp:
                                            self.floatSGrpX = pm.floatSliderGrp(l=self.text["xValue"], cc=pm.Callback(self.floatSGrp_ValueChanged, 0))
                                            self.floatSGrpY = pm.floatSliderGrp(l=self.text["yValue"], cc=pm.Callback(self.floatSGrp_ValueChanged, 1))
                                            self.floatSGrpZ = pm.floatSliderGrp(l=self.text["zValue"], cc=pm.Callback(self.floatSGrp_ValueChanged, 2))
                                        pm.separator()
                                        with pm.horizontalLayout(spacing=4):
                                            pm.button(l=self.text["normalize"], c=pm.Callback(self.buttonNormalize_Clicked))
                                            pm.button(l=self.text["reverse"], c=pm.Callback(self.buttonReverse_Clicked))
                                        
                        with pm.horizontalLayout(spacing=4, height=36):
                            pm.button(l=self.text["runAndClose"], c=pm.Callback(self.buttonRunAndClose_Clicked))
                            pm.button(l=self.text["run"], c=pm.Callback(self.buttonRun_Clicked))
                            pm.button(l=self.text["close"], c=pm.Callback(self.buttonClose_Clicked))
        
        self.buttonAxis_Clicked(0)
    
    def utilFrame_CollapseChanged(self):
        if self.utilFrame.getCollapse():
            self.mainWindow.setHeight(340)
    
    def buttonNormalSize_Clicked(self):
        # Get selected mesh objects.
        transforms = PolyHelper.getSelectedTransforms(forceConvert=True)
        for trans in transforms:
            mesh = trans.getShape()
            mesh.normalSize.set(self.floatSGrpNmlDisp.getValue())
    
    def buttonNormalDisplay_Clicked(self, index):
        # Get selected mesh objects.
        transforms = PolyHelper.getSelectedTransforms(forceConvert=True)
        
        for trans in transforms:
            mesh = trans.getShape()
            if index == 0:
                mesh.displayNormal.set(False)
            else:
                mesh.displayNormal.set(True)
                mesh.normalType.set(index)
    
    def buttonLockNormal_Clicked(self, isOn):
        transforms = pm.selected(type="transform")
        meshes = pm.selected(type="mesh")
        vtxs = PolyHelper.getSelectedVertices(forceConvert=True, extractMesh=True)
        
        for tf in transforms:
            shape = tf.getShape()
            if not shape is None:
                meshes.append(shape)
        
        if isOn:
            if len(meshes) != 0:
                pm.polyNormalPerVertex(meshes, fn=True)
            if len(vtxs) != 0:
                pm.polyNormalPerVertex(vtxs, fn=True)
        else:
            if len(meshes) != 0:            
                pm.polyNormalPerVertex(meshes, ufn=True)
            if len(vtxs) != 0:
                pm.polyNormalPerVertex(vtxs, ufn=True)
    
    def floatSGrp_ValueChanged(self, axisIndex):
        
        if axisIndex == 0:
            self.vector.x = self.floatSGrpX.getValue()
        elif axisIndex == 1:
            self.vector.y = self.floatSGrpY.getValue()
        else:
            self.vector.z = self.floatSGrpZ.getValue()
    
    def buttonGetNormal_Clicked(self):
        
        isWS = self.rdiBtnWorld.getSelect()
        
        # Select algorithm.
        faces = PolyHelper.getSelectedFaces()
        normals = PolyHelper.getFaceNormals(faces, toWorld=isWS)
        if len(normals) == 0:
            vtxs = PolyHelper.getSelectedVertices(forceConvert=True)
            vecs = PolyHelper.getVertexPositions(vtxs)
            # Face no selected.
            if len(vecs) == 0:
                # No selected.
                pm.warning(self.text["noSelectedError"])
                return False
            else:
                # Other than 3 vertices selected.
                normals = PolyHelper.getVertexNormals(vtxs, toWorld=isWS)
                self.vector = Vector3.getAverage(normals).getNormalized()
        else:
            # Face selected.
            self.vector = Vector3.getAverage(normals).getNormalized()
        
        self._setFGrpVector(self.vector)
    
    def buttonAxis_Clicked(self, axisIndex):
        vec = [0.0] * 3
        vec[axisIndex] = 1.0
        self.vector = Vector3(vec)
        self._setFGrpVector(self.vector)
    
    def buttonNormalize_Clicked(self):
        self.vector = self.vector.getNormalized()
        self._setFGrpVector(self.vector)
    
    def buttonReverse_Clicked(self):
        self.vector = self.vector.getReversed()
        self._setFGrpVector(self.vector)
    
    def buttonRunAndClose_Clicked(self):
        if self.buttonRun_Clicked():
            self.buttonClose_Clicked()
    
    def buttonRun_Clicked(self):
        
        isWS = self.rdiBtnWorld.getSelect()
        
        # Get selected.
        vtxFaces = PolyHelper.getSelectedVertexFaces()
        vtxs = PolyHelper.getSelectedVertices(forceConvert=True, fromVertex=True,
                                              fromEdge=True, fromFace=True, fromUV=True)
        
        if len(vtxFaces) == 0 and len(vtxs) == 0:
            pm.warning(self.text["noSelectedError"])
            return False
        
        # Set normals.
        vec = self.vector
        lastShape = ""
        local = vec
        for vtx in (vtxs + vtxFaces):
            if isWS:
                shape = vtx.split(".")[0]
                if lastShape != shape:
                    lastShape = shape
                    matrix = pm.xform(shape, q=True, m=True, ws=True)
                    local = vec.toLocalSpace(matrix)
            pm.polyNormalPerVertex(vtx, x=local.x, y = local.y, z = local.z)

        print "Completed!"
        return True

    def buttonClose_Clicked(self):
        self.mainWindow.delete()
    
    def _getFGrpVector(self):
        x = self.floatSGrpX.getValue()
        y = self.floatSGrpY.getValue()
        z = self.floatSGrpZ.getValue()
        vec = Vector3(x, y, z)
        return vec
    
    def _setFGrpVector(self, vec):
        self.floatSGrpX.setValue(vec.x)
        self.floatSGrpY.setValue(vec.y)
        self.floatSGrpZ.setValue(vec.z)
    

def runMakePlanar(printValue=True, angleThreshold=0.000001):
    MakePlanarToolWindow(printValue, angleThreshold)

def runSetNormal(closeUtil=False):
    SetNormalToolWindow(closeUtil)