# -*- coding: utf-8 -*-

u"""● TrM Blend Normals Tool ver 1.02

2つのメッシュの法線を好きな割合で合成することができます。

▼ インストール

	一般的なPythonスクリプトと同じく、起動時にロードして使用します。
	
	1. mayaスクリプトフォルダにこのファイルを移動します。
        (Windows7 であれば Documents\maya\{バージョン番号}\ja_JP\scripts\ 等）
	2. スクリプトフォルダにuserSetup.pyが無い場合は新規に作成します。
		userSetup.pyは空のテキストファイル(.txt)を作成し、名前をuserSetup.pyに変更します。
		ここで、拡張子が非表示になっていると、userSetup.py.txtになってしまう事があるので注意してください。
		その場合、メモ帳などで開いて別名保存する時に拡張子を変更する事が出来ます。
	3. userSetup.py に以下のコードを追加します。
		import TrM_BlendNormals
	4. Mayaを起動し、以下のコマンドをPythonモードで実行します。
		TrM_BlendNormals.run()
	5. 必要があればコマンドをシェルフに登録してください。

▼ 使い方

    多分ウィンドウを起動すれば分かります。
    例によって、日本語環境以外では自動的に英語に切り替わります。

▼ いろいろ

    ・二次配布の禁止
    ・作者の偽り禁止
    ・使用は自己責任で

▼ 連絡先

    作者名: Tori(仮)
    Twitter: yomogi_k
    Blog: http://kasugaya.blog47.fc2.com/

"""

from itertools import izip
import time
import maya.OpenMaya as om
import pymel.core as pm
import pymel.core.nodetypes as nt
import pymel.core.uitypes as ui


# 言語データ
jpText = {
    "title": u"TrM 法線の合成 ツール",
    "percent": u"追加法線の割合:",
    "sample": u"サンプリング空間:",
    "object": u"オブジェクト",
    "world": u"ワールド",
    "run": u"実行",
    "runAndClose": u"法線の合成",
    "close": u"閉じる",
    "description": u"2つのメッシュを選択してください。\n" +
    u"1. 法線のソース\n" +
    u"2. 設定先",
    "selectionError": u"2つのメッシュを選択してください。",
    "percentError": u"追加法線の割合は0.0に設定できません。",
    "completed": u"法線の合成が完了しました。",
    "cancelled": u"キャンセルされました。",
    "progTitle": u"法線の合成中",
    "progMsg": u"法線を合成しています。"}


enText = {
    "title": u"TrM Blend Normals Tool",
    "percent": u"Percent of the source:",
    "sample": u"Sample space",
    "object": u"Object",
    "world": u"World",
    "run": u"Apply",
    "runAndClose": u"Apply and Close",
    "close": u"Close",
    "description": u"Please select 2 meshes.\n" +
    u"1. Source of Normals to blend.\n" +
    u"2. Destination.",
    "selectionError": u"Please select 2 meshes.",
    "percentError": u"Do not set percent to 0.0.",
    "completed": u"Blend normals completed.",
    "cancelled": u"Cancelled.",
    "progTitle": u"Blend Normals",
    "progMsg": u"Blending..."}


# 環境によって切り替え
lang = pm.about(uiLanguage=True)
if lang == 'ja_JP':
    currentText = jpText
else:
    currentText = enText


def getBasicWindow(w, h, title, btns):

    u"""統一されたデザインのメインウィンドウを提供します。

    :param title: ウィンドウタイトル。
    :type title: str
    :param btns: 下部に配置するボタンのテキスト。
    :type btns: str list
    :returns: 作成されたウィンドウ、レイアウト、ボタンを返します。
              ウィンドウコンテンツはレイアウト上に配置してください。
    :rtype: (pymel.core.uitypes.Window, TabLayout, [Button, ....])
    """

    window = ui.Window(t=title, w=w, h=h)
    formOuter = ui.AutoLayout()

    with formOuter:
        mainTab = ui.TabLayout(tv=False)

        # Bottom buttons.
        buttons = []
        for btn in btns:
            buttons.append(ui.Button(l=btn, h=28, p=formOuter))

    # Attach for the outer layout.
    mrgOut = 2
    mrgBtn = 5
    mrgBtnS = 2
    formOuter.attachForm(mainTab, "top", mrgOut)
    formOuter.attachForm(mainTab, "left", mrgOut)
    formOuter.attachForm(mainTab, "right", mrgOut)

    if buttons:
        formOuter.attachControl(mainTab, "bottom", mrgBtn, buttons[0])

        numBtns = len(buttons)
        width = 100.0 / numBtns
        for i, button in enumerate(buttons):

            ml = mrgBtnS if i != 0 else mrgBtnS + mrgOut
            mr = mrgBtnS if i != numBtns - 1 else mrgBtnS + mrgOut

            formOuter.attachNone(button, "top")
            formOuter.attachForm(button, "bottom", mrgBtn + mrgOut)
            formOuter.attachPosition(button, "left", ml, i * width)
            formOuter.attachPosition(button, "right", mr, (i + 1) * width)

    else:
        formOuter.attachForm(mainTab, "bottom", mrgOut)

    return (window, mainTab, buttons)


def getDagPath(path):
    u"""パス文字列からMDagPathオブジェクトを作成します。"""

    dagPath = om.MDagPath()
    sl = om.MSelectionList()
    sl.add(path)
    sl.getDagPath(0, dagPath)

    return dagPath


def getMeshFn(mesh):
    u"""Meshノードからメッシュ関数セットを取得します。

    :type mesh: pymel.nodetypes.Mesh or Transform or unicode
    :rtype: maya.OpenMaya.MFnMesh
    """

    if isinstance(mesh, (str, unicode)):
        dagPath = getDagPath(mesh)
    else:
        if isinstance(mesh, nt.Transform):
            mesh = mesh.getShape()
        dagPath = getDagPath(mesh.fullPath())

    return om.MFnMesh(dagPath)


def listToMIntArray(intList):
    u"""Python整数リストをMIntArrayへ変換します。"""

    array = om.MIntArray(len(intList))
    setVal = array.set
    for i, val in enumerate(intList):
        setVal(val, i)

    return array


def filterTransform(objects, shapeType=None):
    u"""特定のシェイプを持ったトランスフォームのみを抽出します。

    :param objects: オブジェクトのリスト。
    :type objects: pymel.core.nodetypes.DependNode
    :param shapeType: 指定したシェイプを持つトランスフォームのみを抽出します。
                      タプルで複数指定することもできます。
                      Noneを指定するとシェイプを指定しませんが、
                      type(None)は空のトランスフォーム(グループ)を表します。
    :type shapeType: type or None or tuple
    """

    result = []
    append = result.append
    for obj in objects:
        if isinstance(obj, nt.Transform):
            if shapeType:
                if isinstance(obj.getShape(), shapeType):
                    append(obj)
            else:
                append(obj)
        elif isinstance(obj, shapeType):
            try:
                parent = obj.getParent()
                if isinstance(parent, nt.Transform):
                    append(parent)
            except:
                pass

    return result


def between(theMin, val, theMax):
    u"""最小値と最大値の間で値をトリミングします。"""

    if theMin > val:
        return theMin
    elif theMax < val:
        return theMax
    else:
        return val


def iterGroupByCounts(counts, L):
    u"""指定したカウント毎にサブイテレータを返すジェネレータ。

    :type L: Sliceable sequence.
    :type counts: int list
    """

    idx = 0
    for c in counts:
        nextIdx = idx + c
        yield L[idx:nextIdx]
        idx = nextIdx


def getGroupByCounts(counts, L):
    u"""リストを指定したサイズに分割してグループ化します。

    :type L: Sliceable sequence.
    :type counts: int list
    :rtype: Two-dimensional list
    """

    return [g for g in iterGroupByCounts(counts, L)]


def getNormalIds(meshFn):
    u"""メッシュから頂点の法線参照を取得します。

    :type meshFn: maya.OpenMaya.MFnMesh
    :rtype: list of int list
    :returns: フェース頂点毎の法線を参照する二次元リストです。
              この参照値はMFnMesh.getNormals()で取得した配列に対応しています。

    """

    ids = om.MIntArray()
    counts = om.MIntArray()
    meshFn.getNormalIds(counts, ids)
    normIds = getGroupByCounts(counts, ids)

    return normIds


def getVertexIds(meshFn):
    u"""メッシュから頂点の参照を取得します。

    :type meshFn: maya.OpenMaya.MFnMesh
    :rtype: list of int list
    :returns: フェース頂点毎の頂点座標を参照する二次元リストです。
              この参照値はMFnMesh.getPoints()で取得した配列に対応しています。
    """

    ids = om.MIntArray()
    counts = om.MIntArray()
    meshFn.getVertices(counts, ids)
    vertIds = getGroupByCounts(counts, ids)

    return vertIds


def blendNormals(srcMesh, dstMesh, srcPercent,
                   space=om.MSpace.kPreTransform):

    # プログレスウィンドウを作成
    progTitle = currentText["progTitle"]
    progMsg = currentText["progMsg"]
    pm.progressWindow(title=progTitle, ii=True)
    pm.progressWindow(e=True, pr=0)

    # ブレンド割合
    srcPer = between(0.0, srcPercent, 1.0)
    dstPer = 1.0 - srcPer

    # Mesh API取得
    srcMeshFn = getMeshFn(srcMesh)
    dstMeshFn = getMeshFn(dstMesh)

    # 法線情報を取得
    dstNormals = om.MFloatVectorArray()
    dstMeshFn.getNormals(dstNormals)
    numNormals = dstNormals.length()
    dstNormalIds = getNormalIds(dstMeshFn)
    newNormals = om.MFloatVectorArray(numNormals)

    # 頂点情報を取得
    dstPoints = om.MPointArray()
    dstMeshFn.getPoints(dstPoints)
    dstVertexIds = getVertexIds(dstMeshFn)

    # プログレスバーを初期化
    isCancelled = False
    numDstFaces = len(dstNormalIds)
    pm.progressWindow(e=True, max=numDstFaces, status=progMsg)

    # フェース毎に処理
    for dstFaceId in xrange(numDstFaces):

        dstNormIds = dstNormalIds[dstFaceId]
        dstVertIds = dstVertexIds[dstFaceId]

        # フェースの各頂点毎に処理
        for dstNormId, dstVertId in izip(dstNormIds, dstVertIds):

            # ソースメッシュから最接近法線を取得
            dstPoint = dstPoints[dstVertId]
            srcNor = om.MVector()
            srcMeshFn.getClosestNormal(dstPoint, srcNor, space)

            # 新しい法線を計算
            dstNor = dstNormals[dstNormId]
            newNor = (dstNor * dstPer) + om.MFloatVector(srcNor * srcPer)
            newNormals.set(newNor, dstNormId)

        # 進捗更新
        if dstFaceId % 10 == 0:
            if pm.progressWindow(q=True, isCancelled=True):
                isCancelled = True
                break
            pm.progressWindow(e=True, pr=dstFaceId)

    if not isCancelled:
        # 進捗更新
        pm.progressWindow(e=True, pr=numDstFaces)

        # 送り先メッシュへ法線を設定
        # setNormalsはユーザー法線に設定される為必ずロックが必要
        vertIds = listToMIntArray(range(dstMesh.numVertices()))
        dstMeshFn.lockVertexNormals(vertIds)
        dstMeshFn.setNormals(newNormals)

    # プログレスウィンドウを閉じる
    pm.progressWindow(ep=True)

    return not isCancelled


class mainWindow(object):

    def __init__(self):

        # テキストデータ
        text = currentText

        # ウィンドウを作成
        btns = (text["runAndClose"], text["run"], text["close"])
        result = getBasicWindow(340, 180, text["title"], btns)
        self.window, mainTab, buttons = result
        self.btnRunAndClose, self.btnRun, self.btnClose = buttons

        with mainTab:
            frame = ui.FrameLayout(mw=8, mh=8, lv=False, bv=False)
            with frame:
                with ui.ColumnLayout(adj=True, rs=8):
                    ui.Text(l=text["description"], al="left")

                    self.floatSlider = ui.FloatSliderGrp(l=text["percent"],
                        field=True, max=1.0, w=300, value=0.5, adj=3, pre=3,
                        cat=(1, "right", 16), cw3=(150, 48, 1), co3=(8, 8, 20),
                        ss=0.1, fs=0.1)

                    self.dropDown = ui.OptionMenuGrp(l=text["sample"],
                        adj=2, cw2=(150, 10), cat=(1, "right", 16))

                    with self.dropDown:
                        ui.MenuItem(l=text["object"])
                        ui.MenuItem(l=text["world"])

        # イベントハンドラ設定
        self.btnRun.setCommand(self._btnRunClicked)
        self.btnRunAndClose.setCommand(self._btnRunAndCloseClicked)
        self.btnClose.setCommand(self._btnCloseClicked)

        # 処理間隔の設定
        self.coolTime = 1.0  # 処理終了後に次の処理を受け付けるまでの時間
        self.lastTime = 0

        self.window.show()

        print text["title"]

    def _btnRunClicked(self, *args):

        if time.time() - self.lastTime < self.coolTime:
            return

        sels = pm.selected(o=True)
        sels = filterTransform(sels, nt.Mesh)

        if len(sels) < 2:
            pm.warning(currentText["selectionError"])
            return False

        # パラメータ取得
        per = self.floatSlider.getValue()
        selIdx = self.dropDown.getSelect()
        space = om.MSpace.kWorld if selIdx == 2 else om.MSpace.kObject

        if per == 0.0:
            pm.warning(currentText["percentError"])
            return False

        # ボタンを使用不可にします。(実際には不可になっていないので注意)
        self.btnRun.setEnable(False)
        self.btnRunAndClose.setEnable(False)
        self.btnClose.setEnable(False)
        pm.refresh()

        if blendNormals(sels[0], sels[1], per, space):
            print currentText["completed"]
        else:
            print currentText["cancelled"]

        self.btnRun.setEnable(True)
        self.btnRunAndClose.setEnable(True)
        self.btnClose.setEnable(True)

        self.lastTime = time.time()

        return True

    def _btnRunAndCloseClicked(self, *args):
        if self._btnRunClicked():
            self._btnCloseClicked()

    def _btnCloseClicked(self, *args):
        self.window.delete()

def run():
	mainWindow()