"""Module of slice object helper."""
import sys


def fillDefault(slc):
    """Get a slice to be filled with default values.

    >>> fillDefault(slice(None, 20, None))
    slice(0, 20, 1)

    >>> fillDefault(slice(-5, 10, None))
    slice(-5, 10, 1)

    >>> import sys
    >>> fillDefault(slice(None, None, None)).stop == sys.maxint
    True

    :rtype: slice
    """
    start = slc.start if not slc.start is None else 0
    stop = slc.stop if not slc.stop is None else sys.maxint
    step = slc.step if not slc.step is None else 1

    return slice(start, stop, step)


def trim(slc, length):
    """Get a trimed slice.

    >>> trim(slice(-5, 15), 10)
    slice(0, 10, None)

    >>> trim(slice(-100, -15, 1), 10)
    slice(0, 0, 1)

    :rtype: slice
    """
    start = slc.start
    if start is not None:
        if start < 0:
            start = 0
        elif start > length:
            start = length

    stop = slc.stop
    if stop is not None:
        if stop < 0:
            stop = 0
        elif stop > length:
            stop = length

    return slice(start, stop, slc.step)


def normalize(slc, length, doTrim=False):
    """Get normalized slice.

    >>> normalize(slice(-2), 10)
    slice(None, 8, None)

    >>> normalize(slice(-10, 10), 5)
    slice(-5, 10, None)

    >>> normalize(slice(-10, 10), 5, True)
    slice(0, 5, None)

    :rtype: slice
    """

    start = slc.start
    if start is not None and start < 0:
        start += length

    stop = slc.stop
    if stop is not None and stop < 0:
        stop += length

    slc = slice(start, stop, slc.step)
    if doTrim:
        slc = trim(slc, length)

    return slc


def flip(slc):
    """Get a slice of inverting the start to the stop.

    >>> flip(slice(10, 5, -1))
    slice(6, 11, 1)

    >>> flip(slice(0, 10))
    slice(9, -1, -1)

    >>> flip(slice(None, 5, -1))
    slice(6, None, 1)

    >>> flip(slice(0, 10, 0))
    Traceback (most recent call last):
        ...
    ValueError: Invalid step value.

    :rtype: slice
    """
    start = slc.start
    stop = slc.stop
    step = slc.step

    if step is None:
        step = 1

    if not isinstance(step, int) or step == 0:
        raise(ValueError("Invalid step value."))

    if step > 0:
        temp = start - 1 if start is not None else None
        start = stop - 1 if stop is not None else None
        stop = temp

    elif step < 0:
        temp = start + 1 if start is not None else None
        start = stop + 1 if stop is not None else None
        stop = temp

    return slice(start, stop, step * -1)


if __name__ == "__main__":
    import doctest
    doctest.testmod()
