# -*- coding: utf-8 -*-

u"""Node based Tree Structure.

Ver: 1.01
Auther： Torii
Twitter: yomogi_k
Blog: http://kasugaya.blog47.fc2.com/

License:
    1. Do not delete this docstring.
    2. Free as long as you adhere to the 1.

免責:
    このモジュールを使用して発生した、いかなる損害に対しても
    作者は一切の責任を負いませんので予めご了承ください。

その他:
    作者への連絡はTwitterかブログのコメント欄でお願いします。
"""

import exceptions as ex


class TreeNode(object):
    u"""木構造データのノードを表すクラスです。"""

    separator = "|"

    def __init__(self, name, parent=None, data=None):
        u"""TreeNodeインスタンスを初期化します。

        :param str name: ノード名。
        :param TreeNode parent: 親ノード。
        :param object data: 格納するデータ。
        :raise TypeError: 引数の型が正しくありません。
        :raise NameError: 名前が空文字か、他のノードと競合しています。
        """

        self._name = None
        self._parent = None
        self._data = data
        self._children = {}
        self.name = name  # Check of name format in property setter.
        if not parent is None:
            self.parent = parent  # Append for the parent node.

    def _shiftPath(self, path):
        u"""パス文字列を最初のセクションとそれ以外に分割する。"""

        # パスが文字列ならリストに変換
        if isinstance(path, (str, unicode)):
            path = [name for name in path.split(TreeNode.separator)]
            if path and (path[0] == "" or path[0] == self._name):
                del path[0]

        if not path:
            # path is empty
            return ("", [])
        else:
            shift = path[0]
            del path[0]
            return (shift, path)

    def _checkNameError(self):
        u"""'name'プロパティの内容が正しいかチェックする。"""

        name = self._name
        if not isinstance(name, (str, unicode)):
            msg = u"(TreeNode) このtypeを名前に設定することはできません。" + \
                u" => '%s'" % name.__class__
            raise ex.TypeError(msg)
        if  name == "":
            msg = u"(TreeNode) 空の文字列を名前に使用できません。"
            raise ex.NameError(msg)

    def _checkNodeError(self, parent):
        u"""parentに対して接続した時、整合性が取れているかチェックする。"""

        if not parent is None:
            if not isinstance(parent, TreeNode):
                msg = "(TreeNode) このtypeを親に設定することはできません" + \
                    u" => '%s'" % parent.__class__
                raise ex.TypeError(msg)
            if self._name in parent._children:
                msg = u"(TreeNode) この名前は既に同じ階層で使用されて"\
                    + u"います。 => '%s'" % self._name
                raise NodeError(msg)
            crnt = parent
            while True:
                if crnt == self:
                    msg = u"(TreeNode) 参照が循環しています。" + \
                        u"このノードを親に設定することができません。" + \
                        u" => '%s'" % parent.__class__
                    raise NodeError(msg)
                if crnt.isRoot:
                    break
                crnt = crnt.parent

    def __getitem__(self, path):
        u"""パスを指定してノードを取得します。

        Args:
            path (string): 現在のノードを基準とした相対パス。
                        例 |Foo|Bar|Baz

        Return (TreeNode): 取得したノード
        """

        name, path = self._shiftPath(path)
        if name == "":
            return self
        elif name in self:
            if len(path) == 0:
                return self._children[name]
            else:
                return self._children[name][path]
        else:
            node = "%s['%s']" % (self.name, name)
            msg = u"(TreeNode) 指定されたノードは存在しません。 => %s" % node
            raise NodeError(msg)

    def __delitem__(self, path):
        u"""ノードをツリーから削除します。

        ノードはメモリ上からは削除されないことに注意してください。
        パスについては__getitem__を参照してください。
        """

        node = self[path]
        if not node.isRoot:
            del node.parent._children[node.name]

    def __contains__(self, path):
        u"""指定したパスがツリーに含まれていれば真を返します。

        パスについては__getitem__を参照してください。
        """

        name, path = self._shiftPath(path)
        if name == "":
            return True
        elif name in self._children:
            return True if len(path) == 0 else path in self._children[name]
        else:
            return False

    def __iter__(self):
        u"""親→子の順でノードを取得するイテレータを返します。"""

        yield self
        for child in self._children.values():
            for y in child:
                yield y

    def __reversed__(self):
        u"""子→親の順でノードを取得するイテレータを返します。"""

        for child in reversed(self._children.values()):
            for y in reversed(child):
                yield y
        yield self

    def __len__(self):
        u"""自分自身を含む子ノードの数を返します。"""

        count = 1
        for child in self._children.values():
            count += len(child)
        return count

    def __str__(self):
        u"""このノードの文字列表現を取得します。"""

        return "TreeNode('%s:%s')" % (self._name, str(self._data))

    def __repr__(self):
        u"""全ての子を含むノードの文字列表現を取得します。"""

        if self.isLeaf:
            return "'%s:%s'" % (self._name, repr(self._data))
        else:
            strs = []
            for child in self._children.values():
                strs.append(repr(child))
            return "'%s:%s'{%s}" % \
                (self._name, repr(self._data), ", ".join(strs))

    def addChild(self, childNode):
        u"""他のノードを自分自身の子として接続します。

        :param TreeNode childNode: 追加する子ノード。
        :rtype: TreeNode
        :return: 子ノードをそのまま返します。
        """

        childNode.parent = self
        return childNode

    def create(self, path):
        u"""指定したパスに新しいノードを作成し取得します。

        指定したパスまでの途中のノードが存在しない場合自動的に作成されます。
        既にノードが存在する場合NodeErrorが発生します。

        :param str path: 作成するノードの相対パス。
        :raise TypeError: 引数pathが文字列ではありません。
        :raise NodeError: 既にノードが存在しています。
        :rtype: TreeNode
        :return: 新しく作成したノード。
        """

        name, path = self._shiftPath(path)
        if name == self._name:
            name, path = self._shiftPath(path)

        # 継承先でもインスタンスを作成出来るように
        # __class__メソッドを使ってコンストラクタを呼び出す
        if len(path) == 0:
            return self.__class__(name, self)
        else:
            child = self[name] if name in self else self.__class__(name, self)
            return child.create(path)

    @property
    def isRoot(self):
        u"""このノードがツリーのルートなら真を返します。

        :rtype: bool
        """

        return self._parent == None

    @property
    def isLeaf(self):
        u"""このノードがツリーの末端なら真を返します。

        :rtype: bool
        """

        return len(self._children) == 0

    @property
    def name(self):
        u"""このノードの名前を取得または設定します。

        :rtype: str
        """

        return self._name

    @name.setter
    def name(self, name):
        self._name = name
        self._checkNameError()
        self._checkNodeError(self._parent)

    @property
    def data(self):
        u"""このノードのデータを取得または設定します。

        :rtype: object
        """

        return self._data

    @data.setter
    def data(self, value):
        self._data = value

    @property
    def path(self):
        u"""このノードのルートノードからの相対パスを取得します。

        :rtype: str
        """

        names = []
        crnt = self
        while True:
            print crnt.name, crnt.isRoot
            names.insert(0, crnt.name)
            if crnt.isRoot:
                break
            crnt = crnt.parent
        return TreeNode.separator.join(names)

    @property
    def depth(self):
        u"""このノードのルートノードからの深度を取得します。

        :rtype: int
        :return: このノードがルートノードなら0を返します。"""

        depth = 0
        crnt = self
        while not crnt.isRoot:
            depth += 1
            crnt = crnt.parent
        return depth

    @property
    def parent(self):
        u"""このノードの親ノードを取得または設定します。

        Noneが設定されたとき、親ノードとの接続は解除され自動的に
        ルートノードになります。

        :raise NodeError: 親を変更した事によりノード名が競合している場合、
                          またはノードが循環参照している場合に発生します。
        :rtype: TreeNode
        """

        return self._parent

    @parent.setter
    def parent(self, parent):
        self._checkNodeError(parent)
        self._parent = parent
        parent._children[self.name] = self

    @property
    def root(self):
        u"""このツリーのルートノードを取得します。

        :rtype: TreeNode
        """

        crnt = self
        while not crnt.isRoot:
            crnt = crnt.parent
        return crnt

    @property
    def children(self):
        u"""このノードの子のタプルを取得します。

        :rtype: tuple of TreeNode
        """

        return tuple(self._children.values())

    @property
    def leafs(self):
        u"""このノード以下にある葉のタプルを取得します。

        :rtype: tuple of TreeNode
        """

        nodes = []
        for node in self:
            if node.isLeaf:
                nodes.append(node)
        return tuple(nodes)

    @property
    def brothers(self):
        u"""このノードの兄弟ノードのタプルを取得します。

        :rtype: tuple of TreeNode
        """

        if self.isRoot:
            return (self,)
        else:
            return self.parent.children


class NodeError(ex.StandardError):
    u"""TreeNodeの整合性が取れなくなった時に発生するエラー。"""

    pass
